#!/bin/bash
# This script is compatible with both Bash and Zsh shells.

set -e  # 任何命令失败时终止脚本

# Define target file and directory paths
# 使用 $HOME 变量表示用户主目录，相当于 Windows 的 %USERPROFILE%
CONFIG_FILE="$HOME/.planet-helper.json"
APP_DIR="$HOME/planet-helper"

echo "" # 打印空行
echo "警告: 该操作将删除所有数据，软件和设置信息，请再三确认."
echo "包括:"
echo "- 文件 \"$CONFIG_FILE\""
echo "- 文件夹 \"$APP_DIR\""
echo "" # 打印空行

# Task 1: Warn user and ask for confirmation
# 使用 read -p 进行提示并读取用户输入到 CONFIRM 变量
read -p "按 'y' 继续执行, 按其他按键则终止: " CONFIRM

# 将用户输入转换为小写进行比较
# ${CONFIRM,,} 会将 CONFIRM 变量的值转换为小写 (Bash/Zsh/ksh 语法)
case "${CONFIRM,,}" in
    y)
        echo "" # 打印空行
        echo "开始执行..."
        echo "" # 打印空行

        # Task 2: Check and delete the specific file
        # 使用 -f 检查文件是否存在
        if [ -f "$CONFIG_FILE" ]; then
            echo "删除文件: \"$CONFIG_FILE\" ..."
            # 使用 rm -f 强制删除文件，不提示
            rm -f "$CONFIG_FILE"
            # 检查上一个命令 (rm) 的退出状态，0 表示成功，非 0 表示失败
            if [ $? -ne 0 ]; then
                echo "警告: 无法删除文件 \"$CONFIG_FILE\". 该文件可能正在被使用."
            else
                echo "文件删除成功."
            fi
        else
            echo "文件不存在，跳过."
        fi
        echo "" # 打印空行

        # Task 3: Check and delete the whole directory
        # 使用 -d 检查目录是否存在
        if [ -d "$APP_DIR" ]; then
            echo "删除文件夹: \"$APP_DIR\" ..."
            # 使用 rm -rf 强制递归删除目录及其内容
            rm -rf "$APP_DIR"
            # 检查上一个命令 (rm) 的退出状态
            if [ $? -ne 0 ]; then
                 echo "警告: 无法删除文件夹 \"$APP_DIR\". 某些文件可能正在被使用."
            else
                 echo "文件夹删除成功."
            fi
        else
            echo "文件夹不存在，跳过."
        fi
        echo "" # 打印空行

        echo "执行成功."
        ;; # 结束 case 的 'y' 分支
    *) # 其他任何输入
        echo "" # 打印空行
        echo "程序终止."
        echo "" # 打印空行
        ;; # 结束 case 的其他分支
esac # 结束 case 语句

# Task 4: Press any key to quit
# 使用 read -n 1 -s -r -p 进行提示并等待用户按任意键 (Bash/Zsh 常用)
read -n 1 -s -r -p "按任意键退出..."
echo "" # 打印一个换行，使光标回到下一行

# 使用 exit 0 表示成功退出脚本
exit 0
